package ch.frankel.blog.validation;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertSame;
import static org.junit.Assert.assertTrue;

import java.util.Date;
import java.util.Set;

import javax.validation.ConstraintViolation;
import javax.validation.Validation;
import javax.validation.Validator;

import org.junit.Before;
import org.junit.Test;

import ch.frankel.blog.validation.model.custom.clazz.Person;
import ch.frankel.blog.validation.validator.BirthDeathCoherence;

public class CustomClassValidationTest {

	private Validator validator;

	@Before
	public void setUp() throws Exception {

		validator = Validation.buildDefaultValidatorFactory().getValidator();
	}

	/**
	 * Tests custom validation. Checks there's no violation when the person is
	 * neither born nor dead.
	 */
	@Test
	public void validatePersonNotBornNotDead() {

		Person person = new Person();

		Set<ConstraintViolation<Person>> violations = validator
				.validate(person);

		assertTrue(violations.isEmpty());
	}

	/**
	 * Tests custom validation. Checks there's a unique violation when the person is
	 * dead but not born(!).
	 */
	@Test
	public void validatePersonNotBornButDead() {

		Person person = new Person();

		person.setDeathdate(new Date());
		
		Set<ConstraintViolation<Person>> violations = validator
				.validate(person);

		assertFalse(violations.isEmpty());
		assertEquals(1, violations.size());

		ConstraintViolation<Person> violation = violations.iterator().next();

		assertSame(BirthDeathCoherence.class, violation.getConstraintDescriptor()
				.getAnnotation().annotationType());
	}

	/**
	 * Tests custom validation. Checks there's a unique violation when the person is
	 * dead before being born.
	 */
	@Test
	public void validatePersonDeadBeforeBorn() {

		Person person = new Person();

		person.setDeathdate(new Date(0L));
		person.setBirthdate(new Date());
		
		Set<ConstraintViolation<Person>> violations = validator
				.validate(person);

		assertFalse(violations.isEmpty());
		assertEquals(1, violations.size());

		ConstraintViolation<Person> violation = violations.iterator().next();

		assertSame(BirthDeathCoherence.class, violation.getConstraintDescriptor()
				.getAnnotation().annotationType());
	}
	
	/**
	 * Tests custom validation. Checks there's no violation when the person is
	 * born before being dead.
	 */
	@Test
	public void validatePerson() {

		Person person = new Person();

		person.setDeathdate(new Date());
		person.setBirthdate(new Date(0L));

		Set<ConstraintViolation<Person>> violations = validator
				.validate(person);

		assertTrue(violations.isEmpty());
	}
}
